#!/usr/bin/env python
# -*- coding: utf-8 -*-

###############################################################################
# pci2geotiff_class.py
#
# Auteurs :   Ludovic Granjon, Martin Laloux, redaction@portailsig.org
# Organisme : portailSIG, www.portailsig.org
# Resume :    pci2geotiff http://www.portailsig.org/content/pci2geotiff
#
###############################################################################
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 3 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
# 
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
###############################################################################
 
""" 
géoréférencement de fichiers PCI Raster fournis par la DGI composés d'un tif, GEO_*.tif
et d'un fichier texte, GEO_*.txt 
par Ludovic Granjon et Martin Laloux
version 1
""" 
import os, sys
import glob
from osgeo import gdal
from osgeo import osr
from xml.etree import ElementTree as ET

class tfw:
    """
    Classe tfw
    """
    def __init__(self,fichier):
        self.gcps = []
        self.geotransform = None
        self.fichier = fichier
        self.src_ds = gdal.Open( self.fichier.replace('GEO_','').replace('txt','tif' ))
        self.gcpt = [txt for txt in open(self.fichier, 'r') if "Point " in txt]
    def setGCP(self):
        for index, txt in enumerate(self.gcpt):
            """ 
                création des points d'amer (gdal.GCP)
                découpage des lignes pour obtenir:
                  - les x,y de l'image (pixelXY et GCPLine) 
                  - les x,y géographiques correspondants (coordXY)
            """
            self.gcps.append(gdal.GCP())
            #print index
            pixelXY = txt.split(' ')[0].replace('(','').replace(')','').split(',')
            #print int(pixelXY[0])
            self.gcps[index].GCPPixel = int(pixelXY[0])
     
            self.gcps[index].GCPLine = self.src_ds.RasterYSize-int(pixelXY[1])
     
            coordXY = txt.split(' ')[1].replace('(','').replace(')','').split(',')
            self.gcps[index].GCPX = float(coordXY[0])
            self.gcps[index].GCPY = float(coordXY[1])

    def setGeotransform(self):
        self.geotransform = gdal.GCPsToGeoTransform( self.gcps )

    def writeTfw(self):
        # création du fichier tfw résultant (GEO_*.tfw)
        tfwout = open(self.fichier.replace('GEO_','').replace('txt','tfw' ), 'w')
        # écriture des informations nécessaires, voir http://www.portailsig.org/content/python-georeferencement-et-worldfiles-tfw-jgwpngw
        tfwout.write(str(self.geotransform[1])+'\n')
        tfwout.write(str(self.geotransform[4])+'\n')
        tfwout.write(str(self.geotransform[2])+'\n')
        tfwout.write(str(self.geotransform[5])+'\n')
        tfwout.write(str(self.geotransform[0] + 0.5 * self.geotransform[1] + 0.5 * self.geotransform[2])+'\n')
        tfwout.write(str(self.geotransform[3] + 0.5 * self.geotransform[4] + 0.5 * self.geotransform[5])+'\n')
        # fermeture du fichier tfw
        tfwout.close()

    def writePrj(self,proj):
        # création du fichier prj résultant (GEO_*.tfw)
        prjout = open(self.fichier.replace('GEO_','').replace('txt','prj' ), 'w')
        srs = osr.SpatialReference()
        srs.ImportFromEPSG(proj)
        # écriture des informations nécessaires, voir http://www.portailsig.org/content/python-georeferencement-et-worldfiles-tfw-jgwpngw
        srs.MorphToESRI()
        prjout.write(str(srs.ExportToWkt()))
        # fermeture du fichier prjout
        prjout.close()

    def closeSrc_ds(self):
        self.src_ds = None
            

class projXml:
    """
    Lecture du fichier de config xml
    """
    #On pourra récupérer la liste des projs pour qt avec
    #maClasse.projections.keys()
    #epsg avec
    #maClasse.projections.values()
    def __init__(self, xml):
        self.xml = ET.parse(open(xml,'r'))
        self.projections = {}

    def lireXmlProj(self):
        projs = self.xml.findall('proj')
        for i in projs:
            self.projections[i.attrib['name']]=int(i.text)
            

def pci2geotiff_exe(chemin,proj,xml = 'proj.xml'):
    """
    Traitement d'un repertoire entier
    """
    print chemin

    for nomfich in glob.glob(chemin+'/GEO*.txt'):
        monTfw = tfw(nomfich)
        monTfw.setGCP()
        monTfw.setGeotransform()
        monTfw.writeTfw()
        if proj != None:
            mesProj = projXml(xml)
            mesProj.lireXmlProj()
            epsg = mesProj.projections[proj]           
            monTfw.writePrj(epsg)
        monTfw.closeSrc_ds()

def Usage():
    print """
    Usage : pci2geotiff_class.py [-d chemin_repertoire] [-f chemin_fichier] [-p proj]
    ex pour un répertoire: pci2geotiff_class.py -d ../repci2geotiff/test/ -p Lambert93
    ex pour un fichier : pci2geotiff_class.py -f ../repci2geotiff/test/GEO_8004101515000AB010100011.txt -p Lambert93
    La liste des projections supportées est :
    """
    mesProj = projXml('proj.xml')
    mesProj.lireXmlProj()
    mesProjections = mesProj.projections.keys()
    mesProjections.sort()
    print mesProjections

def main():
    arguments = sys.argv
    if len(arguments) < 2:
        Usage()
    else:
        i = 1
        proj = None
        while i < len(arguments):
            arg = arguments[i]
            if arg == '-h':
                Usage()
                return 1
            elif arg == '-d':
                i+=1
                chemin = arguments[i]
                lot = True
            elif arg == '-f':
                i+=1
                print i
                chemin = arguments[i]
                lot = False
            elif arg == '-p':
                i+=1
                proj = arguments[i]
            elif arg[:1] == '-':
                print "Cette option n'est pas reconnue : ", arg
                Usage()
                return 1            
            i+=1
        if lot:
            try:
                if os.path.exists(chemin):
                    print "traitement par lot de : " +chemin 
                    pci2geotiff_exe(chemin,proj)
                else:
                    print "Le répertoire n'existe pas"
            except:
                print "Le chemin spécifié ne contient pas de fichiers tif et / ou GEO_*.txt"
        else:
            try:
                if os.path.isfile(chemin):
                    print "traitement simple"
                    #traitement d'un fichier
                    monTfw = tfw(chemin)
                    monTfw.setGCP()
                    monTfw.setGeotransform()
                    monTfw.writeTfw()
                    if proj != None:
                        mesProj = projXml('proj.xml')
                        mesProj.lireXmlProj()
                        epsg = mesProj.projections[proj]           
                        monTfw.writePrj(epsg)
                    monTfw.closeSrc_ds()
                else:
                    print "Le fichier n'existe pas"
            except:
                print "Le fichier n'est pas un GEO_*.txt ou est incorrect"

if __name__ == '__main__':
    try:
        main()
    except:
        print """
            Un problème est arrivé,
            Vérifier votre syntaxe avec l'aide ci dessous
            """
        Usage()
